import numpy as np


def ecliptic_to_equatorial(pos_ecliptic, axis_tilt):
    """Rotate position vector from ecliptic to equatorial coordinates"""
    epsilon = np.radians(axis_tilt)
    R = np.array([
        [np.cos(epsilon), 0, np.sin(epsilon)],
        [0, 1, 0],
        [-np.sin(epsilon), 0, np.cos(epsilon)]
    ])
    return R @ pos_ecliptic

def inertial_to_ecef(pos_equatorial, earth_rot_angle_rad):
    """Rotate from inertial equatorial frame to ECEF"""
    R = np.array([
        [ np.cos(earth_rot_angle_rad), np.sin(earth_rot_angle_rad), 0],
        [-np.sin(earth_rot_angle_rad), np.cos(earth_rot_angle_rad), 0],
        [0, 0, 1]
    ])
    return R @ pos_equatorial

def ecef_to_enu(topocentric, lat, lon):
    """Convert ECEF vector to ENU at given lat/lon"""
    sin_lat = np.sin(lat)
    cos_lat = np.cos(lat)
    sin_lon = np.sin(lon)
    cos_lon = np.cos(lon)

    R = np.array([
        [-sin_lon,            cos_lon,           0],
        [-sin_lat*cos_lon, -sin_lat*sin_lon,  cos_lat],
        [ cos_lat*cos_lon,  cos_lat*sin_lon,  sin_lat]
    ])
    return R @ topocentric

def sun_enu_from_earth_ecliptic(
    earth_pos_ecliptic,
    lat_deg, lon_deg, alt_m,
    earth_rot_angle_rad,
    axis_tilt
):
    earth_equatorial = ecliptic_to_equatorial(earth_pos_ecliptic, axis_tilt)

    sun_equatorial = -earth_equatorial

    sun_ecef = inertial_to_ecef(sun_equatorial, earth_rot_angle_rad)

    lat = np.radians(lat_deg)
    lon = np.radians(lon_deg)
    observer_ecef = np.array([0, 0, 0])

    topocentric = sun_ecef - observer_ecef

    print("Coords Topocentric:", topocentric)

    enu = ecef_to_enu(topocentric, lat, lon)

    return enu
